/*
 *
 *  ALL03
 *
 *  26.07.2002
 *
 *  by Fabio Sturman
 *
 *  fabio.sturman@tiscali.it
 *
 *  90slv.c
 *
 *  main module for programming AT90S8515, AT90S/LS2323 and
 *  AT90S/LS2343 in serial mode low voltage
 *
 */

/* Copyright (c) 2002 Fabio Sturman (fabio.sturman@tiscali.it)
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <conio.h>
#include <ctype.h>
#include <string.h>
#include "../include/all03.h"
#include "../include/all03ft.h"
#include "../include/keys.h"
#include "../include/sleep.h"
#include "../include/binmask.h"
#include "attinyhv.h"

/* ATTINY85 - High Voltage Serial Programming pins */
int pin_sii=PIN_MISO_2323;
int pin_sdi=PIN_MOSI_2323;
int pin_sdo=PIN_SCK_2323;
int pin_sci=PIN_CLOCK_2323;
int pin_nreset=PIN_NRESET_2323;
int pin_vcc=PIN_VCC_2323;

char chip_model[64]="ATtiny85";
unsigned short int max_flash_address=0x2000;
unsigned short int max_eeprom_address=0x200;

unsigned short int flash[0x4000]; /* 32k */
unsigned char eeprom[0x400];      /* 1k  */

int lock_bit_1=1;
int lock_bit_2=1;
// ATTiny85
int spien=1;
int selfprgen=1;
int rstdisbl=1;
int dwen=1;
int wdton=1;
int eesave=1;
int bodlevel2=1;
int bodlevel1=1;
int bodlevel0=1;
int ckdiv8=1;
int ckout=1;
int sut1=1;
int sut0=1;
int cksel3=1;
int cksel2=1;
int cksel1=1;
int cksel0=1;

unsigned char signature[3];

char flash_file_name[256]= "flash90.bin";
char eeprom_file_name[256]="eeprom90.bin";


/*- AT90SLV Level 1 --------------------------------------*/

/* write read */
/* as opposed to the Low Voltage programming mode, where only one serial line is bitbanged (excluding the clock line, which is also bit banged),
     in High Voltage programming mode, two lines (SCI, SII) have to be bitbanged before the clock line (SCI) is set to high. 
   not only that, but the HVPM requires a start bit and two more end bits to be sent.
*/
unsigned char wr_rd_serial(unsigned char sdi, unsigned char sii)
{
  int i;
  unsigned char rd;

  rd=0;
  // start bit
  set_sdi(0);
  set_sii(0);
  // toggle serial clock
  set_sci(1);
  if (get_sdo()) rd|=BIT(7); // read MSB of SDO during the clock up of start bit 
  set_sci(0);

  for(i=0; i<8; i++)
  {
    set_sdi(sdi & BIT(7-i) ? 1 : 0);
    set_sii(sii & BIT(7-i) ? 1 : 0);
    set_sci(1);
    if(get_sdo() && i<7) rd|=BIT(6-i); // 6-i as MSB was already read above
    set_sci(0);
  }

  // first end bit
  set_sdi(0);
  set_sii(0);
  // toggle serial clock
  set_sci(1);
  set_sci(0);

  // second end bit
  set_sdi(0);
  set_sii(0);
  // toggle serial clock
  set_sci(1);
  set_sci(0);  

  return(rd);
}

/* put chip in programming mode */
int chip_init(void)
{
  all03_set_gnd_20();

  // sets programming pins to 0
  set_sdi(0);
  set_sii(0);
  set_sdo(0);

  // sets VCC to 5.0V
  all03_set_vcc(5.0);
  // set VHH to 12.0V
  all03_set_vhh(12.0);
  
  // enable VCC output on pin_vcc
  all03_wr_vcc_pin(pin_vcc, 1);
  
  // enable VHH output on RESET PIN
  all03_wr_vhh_pin(pin_nreset, 1);

  // wait 30ms (way too much, it asks for 300uS in the datasheet)
  wait(30.0);

  return(0);
}

void cmd_chip_erase(void)
{
  unsigned char t;

  t=wr_rd_serial(0b10000000, 0b01001100);
  t=wr_rd_serial(0b00000000, 0b01100100);
  t=wr_rd_serial(0b00000000, 0b01101100);

  // wait until SDO goes high for the Chip Erase cycle to finish.
  while(!get_sdo());
}

unsigned char cmd_read_lock_and_fuse_bits(unsigned char sdi1, unsigned char sii1, unsigned char sdi2, unsigned char sii2,
  unsigned char sdi3, unsigned char sii3)
{
  unsigned char t;

  t=wr_rd_serial(sdi1, sii1);
  t=wr_rd_serial(sdi2, sii2);
  t=wr_rd_serial(sdi3, sii3);
  return(t);
}

unsigned char cmd_read_signature_bytes(unsigned short int address)
{
  unsigned char t;
  // pg 160 of ATTiny85 programming manual.
  t=wr_rd_serial(0b00001000, 0b01001100);
  t=wr_rd_serial(0b00000000 | (address & M_0000_0011), 0b00001100);
  t=wr_rd_serial(0b00000000, 0b01101000);
  t=wr_rd_serial(0b00000000, 0b01101100);
  return(t);
}

void cmd_write_default_fuse_bits()
{
  unsigned char t;

  // fuse (low) bits
  t=wr_rd_serial(0b01000000, 0b01001100);

  // A = CKDIV8 Fuse
  // 9 = CKOUT Fuse
  // 8 = SUT1 Fuse
  // 7 = SUT0 Fuse
  // 6 = CKSEL3 Fuse
  // 5 = CKSEL2 Fuse
  // 4 = CKSEL1 Fuse
  // 3 = CKSEL0 Fuse

  //     format is A9876543
  t=wr_rd_serial(0b01100010, 0b00101100); // default values on page 149 of datasheet 
  t=wr_rd_serial(0b00000000, 0b01100100);
  t=wr_rd_serial(0b00000000, 0b01101100);

  // wait until SDO goes high
  while(!get_sdo());

  // fuse high bits

  t=wr_rd_serial(0b01000000, 0b01001100);

  // I = RSTDISBL
  // H = DWEN Fuse 
  // G = SPIEN Fuse
  // F = WDTON Fuse
  // E = EESAVE Fuse
  // D = BODLEVEL2 Fuse
  // C = BODLEVEL1 Fuse
  // B = BODLEVEL0 Fuse

  //     format is IHGFEDCB
  t=wr_rd_serial(0b11011111, 0b00101100); // default values on page 148 of datasheet 
  t=wr_rd_serial(0b00000000, 0b01110100);
  t=wr_rd_serial(0b00000000, 0b01111100);

  // wait until SDO goes high
  while(!get_sdo());

  // extended fuse bits

  t=wr_rd_serial(0b01000000, 0b01001100);
  
  // J = SELFPRGEN Fuse

  //     format is 0000000J
  t=wr_rd_serial(0b00000001, 0b00101100); // default values on page 148 of datasheet 
  t=wr_rd_serial(0b00000000, 0b01100110);
  t=wr_rd_serial(0b00000000, 0b01101110);

  // wait until SDO goes high
  while(!get_sdo());
}

void select_model_tiny85(void)
{
  strcpy(chip_model,"ATtiny85");
  max_flash_address =0x2000;        // 8096 bytes
  max_eeprom_address=0x0200;        // 512 bytes
  pin_sii=PIN_MISO_2323;
  pin_sdi=PIN_MOSI_2323;
  pin_sdo=PIN_SCK_2323;
  pin_nreset=PIN_NRESET_2323;
  pin_vcc=PIN_VCC_2323;
}

void edit_chip_model(void)
{
  int c;

  for(;;)
  {
    clear_screen();
    printf("*** EDIT CHIP MODEL ***\n\n");
    print_chip_model();
    printf("\n");
    printf("A) ATtiny85\n");
    printf("\n");
    printf("Q) QUIT\n");
    printf("\n");
    printf("Enter command: ");
    flush_out();

    c=key_get();
    if((c>=32) && (c<127)) printf("%c\n",c);
    if(islower(c)) c=toupper(c);

    switch(c)
    {
      case 'A': select_model_tiny85(); break;

      case 'Q': return; break;
    }
  }
}

/* LOCK BITS AND FUSES ----------------------------------------------*/

/*
 *  disp_lock_and_fuse_bits()
 *
 *  display lock and fuse bits
 *
 */
void display_lock_and_fuse_bits(void)
{
  if(IS_ATTINY85)
  {
    printf("Fuse and lock bits:\n\n");
    printf("SELFPRGEN=  %d, ",selfprgen ? 1 : 0);
    printf("RSTDISBL=   %d, ",rstdisbl  ? 1 : 0);
    printf("DWEN=       %d, ",dwen  ? 1 : 0);
    printf("SPIEN=      %d\n",spien  ? 1 : 0);
    printf("WDTON=      %d, ",wdton  ? 1 : 0);
    printf("EESAVE=     %d, ",eesave  ? 1 : 0);
    printf("CKDIV8=     %d, ",ckdiv8  ? 1 : 0);
    printf("CKOUT=      %d\n",ckout  ? 1 : 0);
    printf("BODLEVEL2=  %d, ",bodlevel2  ? 1 : 0);
    printf("BODLEVEL1=  %d, ",bodlevel1  ? 1 : 0);
    printf("BODLEVEL0=  %d\n",bodlevel0  ? 1 : 0);
    printf("SUT1=       %d, ",sut1  ? 1 : 0);
    printf("SUT0=       %d\n",sut0  ? 1 : 0);
    printf("CKSEL3=     %d, ",cksel3  ? 1 : 0);
    printf("CKSEL2=     %d, ",cksel2  ? 1 : 0);
    printf("CKSEL1=     %d, ",cksel1  ? 1 : 0);
    printf("CKSEL0=     %d\n",cksel0  ? 1 : 0);
  }
  printf("LOCK BIT 2= %d, ",lock_bit_2 ? 1 : 0);
  printf("LOCK BIT 1= %d\n",lock_bit_1 ? 1 : 0);
}

/*
 *  read_lock_and_fuse_bits()
 *
 *  read lock and fuse bits
 *  from chip
 *
 */
void read_lock_and_fuse_bits(void)
{
  unsigned char fuse;

  clear_screen();
  if(test_if_timer()) return;
  printf("*** READ LOCK AND FUSE BITS ***\n");
  printf(PROC_STRING);
  wait_enter();

  printf("READING...\n");

  if(chip_init()) return;

  if (IS_ATTINY85) {
    // lock bits
    fuse=cmd_read_lock_and_fuse_bits(0b00000100, 0b01001100, 0b00000000, 0b01111000, 0b00000000, 0b01111100);
    lock_bit_2=0;
    lock_bit_1=0;
    if(fuse & BIT(1)) lock_bit_2=1;
    if(fuse & BIT(0)) lock_bit_1=1;
    
    // fuse (low) bits
    fuse=cmd_read_lock_and_fuse_bits(0b00000100, 0b01001100, 0b00000000, 0b01101000, 0b00000000, 0b01101100);
    ckdiv8 = 0;
    ckout = 0;
    sut1 = 0;
    sut0 = 0;
    cksel3 = 0;
    cksel2 = 0;
    cksel1 = 0;
    cksel0 = 0;
    if(fuse & BIT(7)) ckdiv8=1;
    if(fuse & BIT(6)) ckout=1;
    if(fuse & BIT(5)) sut1=1;
    if(fuse & BIT(4)) sut0=1;
    if(fuse & BIT(3)) cksel3=1;
    if(fuse & BIT(2)) cksel2=1;
    if(fuse & BIT(1)) cksel1=1;
    if(fuse & BIT(0)) cksel0=1;

    // fuse high bits
    fuse=cmd_read_lock_and_fuse_bits(0b00000100, 0b01001100, 0b00000000, 0b01111010, 0b00000000, 0b01111110);
    rstdisbl = 0;
    dwen = 0;
    spien = 0;
    wdton = 0;
    eesave = 0;
    bodlevel2 = 0;
    bodlevel1 = 0;
    bodlevel0 = 0;
    if(fuse & BIT(7)) rstdisbl=1;
    if(fuse & BIT(6)) dwen=1;
    if(fuse & BIT(5)) spien=1;
    if(fuse & BIT(4)) wdton=1;
    if(fuse & BIT(3)) eesave=1;
    if(fuse & BIT(2)) bodlevel2=1;
    if(fuse & BIT(1)) bodlevel1=1;
    if(fuse & BIT(0)) bodlevel0=1;

    // extended fuse bits
    fuse=cmd_read_lock_and_fuse_bits(0b00000100, 0b01001100, 0b00000000, 0b01101010, 0b00000000, 0b01101110);
    selfprgen = 0;
    if(fuse & BIT(0)) selfprgen=1;
  }

  all03_init_h();

  display_lock_and_fuse_bits();
  printf("\nOK\n");
}

/* ERASE -----------------------------------------------------------*/

/*
 *  chip_erase()
 *
 *  erase flash, eeprom and lock bits
 *  fuses unchanged
 *
 */
void chip_erase(void)
{
  clear_screen();
  if(test_if_timer()) return;
  printf("** ERASE ***\n");
  printf(PROC_STRING);
  wait_enter();

  printf("ERASING...\n");
  if(chip_init()) return;

  cmd_chip_erase();

  all03_init_h();

  printf("OK\n");
}

/*
 *  write_default_fuse_bits()
 *
 *  Writes the default Fuse bits for ATtiny85
 *  
 *
 */
void write_default_fuse_bits(void)
{
  clear_screen();
  if(test_if_timer()) return;
  printf("** WRITE DEFAULT FUSE BITS ***\n");
  printf(PROC_STRING);
  wait_enter();

  printf("WRITING...\n");
  if(chip_init()) return;

  cmd_write_default_fuse_bits();

  all03_init_h();

  printf("OK\n");
}


/*
 *  read_signature_bytes()
 *
 *  reads signature bytes
 *
 *  ATTiny85
 *  SIG 0 - 0x1E manifactured by Atmel
 *  SIG 1 - 0x93 8k flash
 *  SIG 2 - 0x0B indicates ATtiny85
 *
 */
void read_signature_bytes(void)
{
  int i;

  clear_screen();
  if(test_if_timer()) return;
  printf("*** READ SIGNATURE ***\n");
  printf(PROC_STRING);
  wait_enter();

  printf("READING...\n");
  if(chip_init()) return;

  for(i=0; i<3; i++)
  {
    signature[i]=cmd_read_signature_bytes(i);
  }

  all03_init_h();

  for(i=0;i<3;i++)
    printf("Signature[%d]=%02x\n", i, signature[i]);

  printf("OK\n");
}

/* displays help screen */
void display_help(void)
{
    printf("use: attinyhv [<option>] [<file_name>]\n");
    printf("option:\n");
    printf(" -s       skip processor speed test\n");
    printf(" -h       display this screen\n");
    printf(" -bxxx    set all03 base address\n");
    printf(" -ttiny85 select ATtiny85\n");

    exit(1);
}

/* main program */
int main(int argc, char *argv[])
{
  int c, ac, skip, tbase;

  all03_init_h();
  disable_ctrl_c_break();

  ac=argc-1;
  skip=0;
  while(ac)
  {
    if(!strcmp(argv[ac],"-s") || !strcmp(argv[ac],"-S")) skip=1;
    else if(!strcmp(argv[ac],"-h") || !strcmp(argv[ac],"-H"))
    {
      display_help();
      return(1);
    }
    else if(!strncmp(argv[ac],"-b",2) || !strncmp(argv[ac],"-B",2))
    {
      sscanf(&argv[ac][2],"%x",&tbase);
      all03_set_base_address(tbase);
    }
    else if(!stricmp(argv[ac],"-ttiny85")) select_model_tiny85();
    else strcpy(flash_file_name,argv[ac]);
    ac--;
  }

  if(!skip)
  {
    printf("TESTING SPEED...\n");
    if(!test_timer()) printf("TIMER OK\n");
    else printf("\n\007* ALL03 Error - Timing not OK!\n");
  }

  if(all03_test_base_address())
  {
    printf("\n\007* ALL03 Error - Programmer not found at base address %04x!\n",all03_get_base_address());
    printf("Press <ENTER>\n");
    wait_enter();
  }

  for(;;)
  {
    clear_screen();
    printf("-----------------------------------------------------------------\n");
    printf("-- ALL03 - ATTINY SERIAL HIGH VOLTAGE PROGRAMMING MODULE --------\n");
    printf("----------------------------------by FaSt - Ver 1.0BETA3---------\n");
    printf("\n");
    print_chip_model();
    printf("\n");
    printf("    Menu:\n");
    printf("\n");
    printf("6 - Edit base address\n");
    printf("L - Read fuse & lock bits\n");
    printf("S - Read signature\n");
    printf("E - Chip Erase\n");
    printf("X - Write default Fuse values\n");
    printf("T - Select chip model\n");
    //printf("{ - Test\n");
    //printf("} - Dump ALL03 registers\n");
    printf("Q - quit\n");
    printf("\n");
    printf("Enter command: ");
    flush_out();

    c=key_get();
    if((c>=32) && (c<=126)) printf("%c\n",c);
    if(islower(c)) c=toupper(c);

    switch(c)
    {
      case '6': all03_edit_base(); break;

      case 'S': read_signature_bytes(); break;
      case 'L': read_lock_and_fuse_bits(); break;
      case 'E': chip_erase(); break;
      case 'T': edit_chip_model(); break;
      case 'X': write_default_fuse_bits(); break;

      case '{': test(); break;
      case '}': all03_dump(); break;

      case 'Q':
        enable_ctrl_c_break();
        return(0);
      break;
    }

    printf("Press <ENTER>\n");
    wait_enter();
  }
  return(0);
}
